/**
 * An adaptor, transforming the JDBC interface to the TableModel interface.
 *
 * @version 1.20 09/25/97
 * @author Philip Milne
 */

import java.util.Vector;
import java.sql.*;
import javax.swing.table.AbstractTableModel;
import javax.swing.event.TableModelEvent;

public class JDBCAdapter extends AbstractTableModel {
    Connection          connection;
    PreparedStatement   statement;
    ResultSet           resultSet;
    String[]            columnNames = {};
    Vector				rows = new Vector();
    ResultSetMetaData   rsmd;
	private String 		tablename;

    public JDBCAdapter(String url, String driverName,
                       String user, String passwd) {
        try {
			DriverManager.registerDriver( new oracle.jdbc.driver.OracleDriver());
           	Class.forName(driverName);
            System.out.println("Opening db connection");

            connection = DriverManager.getConnection(url, user, passwd);
        }
        catch (ClassNotFoundException ex) {
            System.err.println("Cannot find the database driver classes.");
            System.err.println(ex);
        }
        catch (SQLException ex) {
            System.err.println("Cannot connect to this database.");
            System.err.println(ex);
        }
     }

    public void executeQuery(String query) {
        if (connection == null) {
            System.err.println("There is no database to execute the query.");
            return;
        }
        try {
            statement = connection.prepareStatement(query);
            resultSet = statement.executeQuery();
            rsmd = resultSet.getMetaData();
			/** ---------------------------------------------*/
			/* a bad way to get tablename but only works for a single tablename
			 * can be improved to get more tablenames */
			Keywords2UpperCase k2 = new Keywords2UpperCase();
			query = k2.keytoUpperCase(query);
		    System.out.println("\n" + query + "\n");

			int i1 = query.indexOf("FROM") + 5;
			int i2 = query.indexOf("WHERE");
			if (i2 == -1) i2 = query.length();
			tablename = query.substring(i1,i2);
			System.out.println("Table name :" + tablename);
			/** -----------------------------------------------*/
            int numberOfColumns =  rsmd.getColumnCount();
            columnNames = new String[numberOfColumns];
            // Get the column names and cache them.
            for(int column = 0; column < numberOfColumns; column++) {
                columnNames[column] = rsmd.getColumnLabel(column + 1);
            }
            // Get all rows.
            rows = new Vector();
            while (resultSet.next()) {
                Vector newRow = new Vector();
                for (int i = 1; i <= numberOfColumns; i++) {
	            	newRow.addElement(resultSet.getObject(i));
                }
                rows.addElement(newRow);
            }
            fireTableChanged(null); // Tell the listeners a new table has arrived.
        }
        catch (SQLException ex) {
            System.err.println(ex);
        }
    }

    public void close() throws SQLException {
        System.out.println("Closing db connection");
        resultSet.close();
        statement.close();
        connection.close();
    }

    protected void finalize() throws Throwable {
        close();
        super.finalize();
    }

    //////////////////////////////////////////////////////////////////////////
    //
    //             Implementation of the TableModel Interface
    //
    //////////////////////////////////////////////////////////////////////////

    // TableModel has these methods defined abstractly .

    public String getColumnName(int column) {
        if (columnNames[column] != null) {
            return columnNames[column];
        } else {
            return "";
        }
    }

    public Class getColumnClass(int column) {
        int type;
        try {
            type = rsmd.getColumnType(column+1);
        }
        catch (SQLException e) {
            return super.getColumnClass(column);
        }

        switch(type) {
        case Types.CHAR:
        case Types.VARCHAR:
        case Types.LONGVARCHAR:
            return String.class;

        case Types.BIT:
            return Boolean.class;

        case Types.TINYINT:
        case Types.SMALLINT:
        case Types.INTEGER:
            return Integer.class;

        case Types.BIGINT:
            return Long.class;

        case Types.FLOAT:
        case Types.DOUBLE:
            return Double.class;

        case Types.DATE:
            return java.sql.Date.class;

        default:
            return Object.class;
        }
    }

    public boolean isCellEditable(int row, int column) {
        try {
            return rsmd.isWritable(column + 1);
        }
        catch (Exception e) {
            return false;
        }
    }

	// JTable uses this method to determine how manycolumns to be created and displayed on initialization
    public int getColumnCount() {
        return columnNames.length; 
    }

    // Data methods

	// JTable uses this method to determine how mant rows to be created and displayed
	// called frequently
    public int getRowCount() {
        return rows.size();
    }

    public Object getValueAt(int aRow, int aColumn) {
        Vector row = (Vector)rows.elementAt(aRow);
        return row.elementAt(aColumn);
    }

    public String dbRepresentation(int column, Object value) {
        int type;

        if (value == null) {
            return "null";
        }

        try {
            type = rsmd.getColumnType(column+1);
        }
        catch (SQLException e) {
            return value.toString();
        }

        switch(type) {
        case Types.INTEGER:
        case Types.DOUBLE:
        case Types.FLOAT:
            return value.toString();
        case Types.BIT:
            return ((Boolean)value).booleanValue() ? "1" : "0";
        case Types.DATE:
            return value.toString(); // This will need some conversion.
        default:
            return "\'"+value.toString()+"\'";
        }
    }

    public void setValueAt(Object value, int row, int column) {
        try {
            //String tableName = rsmd.getTableName(column+1);
            // Some of the drivers seem buggy, tableName should not be null.

            if (tablename.equals("")) {
                System.out.println("Table name returned null.");
				//System.exit(0);
            } else {
            String columnName = getColumnName(column);
            String query =
                "update "+tablename+
                " set "+columnName+" = "+dbRepresentation(column, value)+
                " where ";
            // We don't have a model of the schema so we don't know the
            // primary keys or which columns to lock on. To demonstrate
            // that editing is possible, we'll just lock on everything.
            for(int col = 0; col<getColumnCount(); col++) {
                String colName = getColumnName(col);
                if (colName.equals("")) {
                    continue;
                }
                if (col != 0) {
                    query = query + " and ";
                }
                query = query + colName +" = "+
                    dbRepresentation(col, getValueAt(row, col));
            }
            System.out.println(query);
            System.out.println(" sending update to database");
            statement = connection.prepareStatement(query);
            statement.executeUpdate();
            System.out.println(" Database Updated successfully");
			}
        }
        catch (SQLException e) {
            System.err.println("Update failed");
            e.printStackTrace();
        }
        Vector dataRow = (Vector)rows.elementAt(row);
        dataRow.setElementAt(value, column);
    }
}
