package sat;

import java.sql.*;
import java.util.*;

/** 
 * This is the implementation of the JDBC query engine. It processes the
 * transactions to the database. It has various methods like 
 * <code>getPassword</code>, etc which return the appropriate info from
 * the database. It has two basic functions:
 * <ol>
 * <li> Retrieving information from the database
 * <li> Inserting information into the database
 *	</ol>
 *
 * @author	Satyen Kale
 * @version	1.00, 26/9/2K
 */

public class QueryEngine {
	/* should this be package access? 
	 * maybe useful later
	 */
	Connection con;
	DatabaseMetaData dbmd;
	
/** A constructor for the class. Initialises the database connection.
 * Creates instances of Statement, ResultSet, ResultSetMetaData classes.
 */
	public QueryEngine() {
		System.out.println("Connecting to the Oracle database, please wait ...");
		try {
			DriverManager.registerDriver( new oracle.jdbc.driver.OracleDriver());
			con = DriverManager.getConnection(Config.connection, Config.loginid, Config.password);
			dbmd = con.getMetaData();
		} catch (SQLException sqle) {
			System.err.println("Unable to connect. Exiting...");
			System.exit(1);
		}
	}

	/** gets a password for a given userid.
	*
	* @param	userid the user login id
	* @return	the password of the user
	*/
	public synchronized String getPassword(String userid) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery("select password from passwords where login_id = '" + userid + "'");
			String ret = (rs.next() ? rs.getString(1) : null);
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
			
		return null;
	}

	/**
	 * returns the event name for given event_id
	 * 
	 * @param event_id the ID of the event
	 * @return the name of the event
	 */
	public synchronized String getEventName(String event_id) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery("select event_name from Event where event_id = '" + event_id + "'");
			String ret = (rs.next() ? rs.getString(1) : null);
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
			
		return null;
	}

	/** returns a <code>Vector</code> of (subevent name, <code>TimeInfo</code>)
	 * for the schedule for an entire event.
	 *
	 * @param event_id the ID of the event
	 * @return a <code>Vector</code> of the required data.
	 */
	public synchronized Vector getEventSchedule(String event_id) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery(
				" select subevent.subevent_name, day, start_time, end_time " +
				" from SubEvent, Occurs_On " +
				" where SubEvent.event_id = '" + event_id + "' and " +
				" SubEvent.subevent_id = Occurs_On.subevent_id and " +
				" Occurs_On.event_id = '" + event_id + "' " +
				" order by SubEvent.subevent_name, day, start_time ");
			
			Vector ret = new Vector(10, 10);
			TimeInfo ti;
			
			while (rs.next()) {
				ti = new TimeInfo();
				ti.set(rs.getString(2), rs.getString(3), rs.getString(4));
				ret.addElement(rs.getString(1));
				ret.addElement(ti);
			}
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
			
		return null;
	}
				
			 
	/**
	 * returns a <code>Vector</code> of login_id, name of flos registered
	 * for an event.
	 *
	 * @param event_id the ID of the event
	 * @return a <code>Vector</code> of the required data.
	 */
	public synchronized Vector getEventParticipants(String event_id) {
		try {
			String table = "Individual";
			if (SharedObjects.val.isTeamBased(event_id)) table = "Contingent";
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery(
				" select registered_for.login_id, name " +
				" from " + table + ", registered_for " +
				" where event_id = '" + event_id + "' and " +
				" registered_for.login_id = " + table + ".login_id and " + 
				" registered_for.approved = 1");
			Vector ret = new Vector(10, 10);
			while (rs.next()) {
				ret.addElement(rs.getString(1));
				ret.addElement(rs.getString(2));
			}
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
			
		return null;
	}
	
	/**
	 * returns a <code>Vector</code> of login_id, name of flos participating
	 * in a subevent.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @return a <code>Vector</code> of the required data.
	 */
	public synchronized Vector getSubEventParticipants(String event_id, String subevent_id) {
		try {
			String table = "Individual";
			String name = "name";
			if (SharedObjects.val.isTeamBased(event_id)) { table = "Team"; name = "team_name"; }
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery(
				" select " + table + ".login_id, " + name +
				" from " + table + ", participates_in " +
				" where event_id = '" + event_id + "' and " +
				" subevent_id = '" + subevent_id + "' and " +
				" participates_in.pid = " + table + ".pid and " +
				" approved = 1");
			Vector ret = new Vector(10, 10);
			while (rs.next()) {
				ret.addElement(rs.getString(1));
				ret.addElement(rs.getString(2));
			}
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
			
		return null;
	}

	/**
	 * returns the subevent name for given subevent_id
	 * 
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @return the name of the subevent
	 */
	public synchronized String getSubEventName(String event_id, String subevent_id) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery("select subevent_name from SubEvent where event_id = '" + event_id + "' and subevent_id = '" + subevent_id + "'");
			String ret = (rs.next() ? rs.getString(1) : null);
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
			
		return null;
	}
	
	/** gets the email ID of an Individual/Contingent registering
	 * for an event.
	 *
	 * @param event_id the ID of the event
	 * @param login_id the login ID
	 * @return the email ID
	 */
	public synchronized String getEventEmailID(String event_id, String login_id) {
		try {
			String table = "Individual";
			if (SharedObjects.val.isTeamBased(event_id)) table = "Contingent";
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery(
				" select email_id from " + table + ", registered_for " +
				" where event_id = '" + event_id + "' and " +
				" registered_for.login_id = '" + login_id + "' and " +
				table + ".login_id = '" + login_id + "'");
			String ret = (rs.next() ? rs.getString(1) : null);
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
			
		return null;
	}
	
	/** gets the email ID of an Individual/Team participating
	 * in a subevent.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @param login_id the login ID
	 * @param name the name of the participant
	 * @return the email ID
	 */
	public synchronized String getSubEventEmailID(String event_id, String subevent_id, String login_id, String name) {
		try {
			String table = "Individual";
			boolean IC_flag = false;
			if (SharedObjects.val.isTeamBased(event_id)) { table = "Team"; IC_flag = true; }
			int pid = getPID(login_id, name, IC_flag);
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery(
				" select email_id from " + table + ", participates_in " +
				" where event_id = '" + event_id + "' and " +
				" subevent_id = '" + subevent_id + "' and " +
				" participates_in.pid = '" + pid + "' and " +
				table + ".pid = '" + pid + "'");
			String ret = (rs.next() ? rs.getString(1) : null);
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
			
		return null;
	}
		
	/**
	 * get email ID of event manager.
	 *
	 * @param event_id the ID of the event
	 * @return the email ID
	 */
	public synchronized String getEMEmail(String event_id) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery(
				" select email_id from Event_Manager where event_id = '" + event_id + "'"); 
			String ret = (rs.next() ? rs.getString(1) : null);
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
			
		return null;
	}
	
	/**
	 * get email ID of subevent manager.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @return the email ID
	 */
	public synchronized String getSEMEmail(String event_id, String subevent_id) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery(
				" select email_id from SubEvent_Manager where event_id = '" + event_id + "' and subevent_id = '" + subevent_id + "'"); 
			String ret = (rs.next() ? rs.getString(1) : null);
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
			
		return null;
	}
				
	/**
	 * returns the SEM login id for a given SubEvent
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the SubEvent
	 * @return the login ID of the subevent manager
	 */
	public synchronized String getSEMLogin(String event_id, String subevent_id) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery("select login_id from SubEvent_Manager where event_id = '" + event_id + "' and subevent_id = '" + subevent_id + "'");
			String ret = (rs.next() ? rs.getString(1) : null);
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
			
		return null;
	}
			
	/**
	 * returns the team size for a given subevent.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @return team size
	 */
	public synchronized int getTeamSize(String event_id, String subevent_id) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery("select team_size from SubEvent where event_id = '" + event_id + "' and subevent_id = '" + subevent_id + "'");
			int ret = (rs.next() ? rs.getInt(1) : -1);
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
			
		return -1;
	}

	/**
	 * changes a password for a given user.
	 *
	 * @param	userid	the user login id
	 * @param	newpass	the new password
	 * @return	true on success, false on failure
	 */

	public synchronized boolean changePassword(String userid, String newpass) {
		if (newpass.length() > 8) return false;
		try {
			con.setAutoCommit(true);
			Statement stmt = con.createStatement();
			stmt.executeUpdate("update passwords set password = '" + newpass + "' where login_id = '" + userid + "'");
			stmt.close();
			return true;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
			 
		return false;
	}

	/**
	 * returns the capacity for an event.
	 * 
	 * @param event_id the ID of the event
	 * @return the capacity of the event
	 */
	 public synchronized int getCapacity(String event_id) {
		 try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery(
				"select capacity from Event where event_id = '" + event_id + "'");
			if (! rs.next()) { rs.close(); stmt.close(); return -1; }
			int ret = rs.getInt(1);
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return -1;
	}

	/**
	 * returns a <code>Vector</code> of (login_id, name) of flos whose registration
	 * for the event is still pending approval of the event manager.
	 *
	 * @param event_id the ID of the event
	 * @return a <code>Vector</code> of the necessary data
	 */
	public synchronized Vector getPending(String event_id) {
		try {
			Statement stmt = con.createStatement();
			String table = "Individual";
			if (SharedObjects.val.isTeamBased(event_id)) table = "Contingent";
			ResultSet rs = stmt.executeQuery(
				" select registered_for.login_id, name  from registered_for, " + table + 
				" where event_id = '" + event_id + "' and approved = 0 and registered_for.login_id = " + table + ".login_id");

			Vector ret = new Vector(10, 10);
			
			while (rs.next()) {
				ret.addElement(rs.getString(1));
				ret.addElement(rs.getString(2));
			}
			
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return null;
	}
		
	/**
	 * returns a <code>Vector</code> of (login_id, name) of flos whose participation
	 * in the SubEvent is still pending approval of the event manager.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @return a <code>Vector</code> of the necessary data
	 */
	public synchronized Vector getPending(String event_id, String subevent_id) {
		try {
			Statement stmt = con.createStatement();
			String table = "Individual";
			String prefix = "";
			if (SharedObjects.val.isTeamBased(event_id)) { table = "Team"; prefix = "team_"; }
			ResultSet rs = stmt.executeQuery(
				" select login_id, " + prefix + "name  from Participates_in, " + table + 
				" where event_id = '" + event_id + "' and " +
				" subevent_id = '" + subevent_id + "' and " +
				" approved = 0 and Participates_in.pid = " + table + ".pid");

			Vector ret = new Vector(10, 10);
			
			while (rs.next()) {
				ret.addElement(rs.getString(1));
				ret.addElement(rs.getString(2));
			}
			
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return null;
	}

	/**
	 * returns a <code>Vector</code> of <code>Judge</code> objects 
	 * for a given SubEvent.
	 * 
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the SubEvent
	 * @return a <code>Vector</code> of the <code>Judge</code> objects
	 */
	public synchronized Vector getJudges(String event_id, String subevent_id) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery(
				" select Judge.email_id, name, phone, address1, address2, fax " +
				" from Judge, Judges " +
				" where Judges.email_id = Judge.email_id and" +
				" event_id = '" + event_id + "' and " +
				" subevent_id = '" + subevent_id + "'");

			Vector ret = new Vector(10, 10);
			Judge j;
			while (rs.next()) {
				j = new Judge();
				j.email_id = rs.getString(1);
				j.name = rs.getString(2);
				j.phone = rs.getString(3);
				j.address1 = rs.getString(4);
				j.address2 = rs.getString(5);
				j.fax = rs.getString(6);
				ret.addElement(j);
			}

			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return null;
	}
			 
	 /**
	 * returns an <code>IndData</code> for an individual partcipant
	 *
	 * @param userid the login ID of the Individual
	 * @param event_id the id of the event
	 * @return <code>IndData</code> of the required info
	 */
	public synchronized IndData getIndData(String event_id, String userid) {
		try {
			Statement stmt = con.createStatement();
			IndData id = new IndData();
			ResultSet rs = stmt.executeQuery(
				" select email_id, name from Individual where login_id = '" + userid + "'");
			if (! rs.next()) { rs.close(); stmt.close(); return null; }
			id.email = rs.getString(1);
			id.name = rs.getString(2);
			rs = stmt.executeQuery(
				"select subevent.subevent_name, participates_in.subevent_id  " +
				" from participates_in, Individual, subevent " +
				" where Individual.login_id = '" + userid + "'" +
				" and subevent.event_id = '" + event_id + "'" +
				" and participates_in.pid = Individual.pid " +
				" and participates_in.subevent_id = subevent.subevent_id");
			while (rs.next()) {
				id.insertSE(rs.getString(1), rs.getString(2), true);
			}
			rs = stmt.executeQuery(
				" select subevent_name, subevent_id " +
				" from subevent " +
				" where event_id = '" + event_id + "'" +
				" minus " +
				"select subevent.subevent_name, participates_in.subevent_id  " +
				" from participates_in, Individual, subevent " +
				" where Individual.login_id = '" + userid + "'" +
				" and subevent.event_id = '" + event_id + "'" +
				" and participates_in.pid = Individual.pid " +
				" and participates_in.subevent_id = subevent.subevent_id");
			while (rs.next()) {
				id.insertSE(rs.getString(1), rs.getString(2), false);
			}

			rs = stmt.executeQuery(
				" select attrib, value from EM_SPEC_PARTICIPANT_ATTRIB_VAL " +
				" where event_id = '" + event_id + "' and login_id = '" + userid + "'");
			while (rs.next()) {
				id.insertEMSpecPVals(rs.getString(1), rs.getString(2));
			}
			rs.close();
			stmt.close();
			return id;
		} catch (SQLException sqle) {
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return null;
	}
	
	/**
	 * returns an <code>ContData</code> for a Contingent
	 *
	 * @param userid the login ID of the Contingent
	 * @param event_id the id of the event
	 * @return <code>ContData</code> of the required info
	 */
	public synchronized ContData getContData(String event_id, String userid) {
		try {
			Statement stmt = con.createStatement();
			ContData cd = new ContData();
			ResultSet rs = stmt.executeQuery(
				" select email_id, name, cl_name from Contingent where login_id = '" + userid + "'");
			if (! rs.next()) { rs.close(); stmt.close(); return null; }
			cd.email = rs.getString(1);
			cd.name = rs.getString(2);
			cd.cl_name = rs.getString(3);
			rs = stmt.executeQuery(
				" select subevent.subevent_name, participates_in.subevent_id, team.team_name  " +
				" from participates_in, Team, subevent " +
				" where Team.login_id = '" + userid + "'" +
				" and subevent.event_id = '" + event_id + "'" +
				" and participates_in.pid = Team.pid " +
				" and participates_in.subevent_id = subevent.subevent_id");
			while (rs.next()) {
				cd.insertSE(rs.getString(1), rs.getString(2), rs.getString(3));
			}
			rs = stmt.executeQuery(
				" select subevent_name, subevent_id " +
				" from subevent " +
				" where event_id = '" + event_id + "'" +
				" minus " +
				" select subevent.subevent_name, participates_in.subevent_id " +
				" from participates_in, Team, subevent " +
				" where Team.login_id = '" + userid + "'" +
				" and subevent.event_id = '" + event_id + "'" +
				" and participates_in.pid = Team.pid " +
				" and participates_in.subevent_id = subevent.subevent_id");
			while (rs.next()) {
				cd.insertSE(rs.getString(1), rs.getString(2), null);
			}
			rs = stmt.executeQuery(
				" select attrib, value from EM_SPEC_PARTICIPANT_ATTRIB_VAL " +
				" where event_id = '" + event_id + "' and login_id = '" + userid + "'");
			while (rs.next()) {
				cd.insertEMSpecPVals(rs.getString(1), rs.getString(2));
			}
			rs.close();
			stmt.close();
			return cd;
		} catch (SQLException sqle) {
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return null;
	}

	/**
	 * returns an <code>IndSEData</code> object for a given individual 
	 * participant in a given subevent.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @param login_id the login id of the individual
	 * @return a <code>IndSEData</code> object
	 */
	public synchronized IndSEData getIndSEData(String event_id, String subevent_id, String login_id) {
		try {
			Statement stmt = con.createStatement();
			IndSEData id = new IndSEData();
			ResultSet rs = stmt.executeQuery(
				" select pid, email_id, name from Individual where login_id = '" + login_id + "'");
			if (! rs.next()) { rs.close(); stmt.close(); return null; }
			int pid = rs.getInt(1);
			id.email = rs.getString(2);
			id.name = rs.getString(3);
			id.pid = pid;

			rs = stmt.executeQuery(
				" select attrib, value from SEM_SPEC_PARTICIPANT_ATTRIB_VL " +
				" where event_id = '" + event_id + "' and " +
				" subevent_id = '" + subevent_id + "' and " +
				" pid = " + pid);
			
			while (rs.next()) {
				id.insertSEMSpecPVals(rs.getString(1), rs.getString(2));
			}
			rs.close();
			stmt.close();
			return id;
		} catch (SQLException sqle) {
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return null;
	}
	
	/**
	 * returns an <code>TeamSEData</code> object for a given team 
	 * participant in a given subevent.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @param login_id the login id of the Contingent
	 * @return a <code>IndSEData</code> object
	 */
	public synchronized TeamSEData getTeamSEData(String event_id, String subevent_id, String login_id) {
		try {
			Statement stmt = con.createStatement();
			TeamSEData td = new TeamSEData();
			ResultSet rs = stmt.executeQuery(
				" select Team.pid, Team.name, Team.email_id, Team.team_name  from Team, participates_in " +
				" where login_id = '" + login_id + "' and " +
				" Team.pid = participates_in.pid and " +
				" event_id = '" + event_id + "' and " +
				" subevent_id = '" + subevent_id + "'");
			if (! rs.next()) { rs.close(); stmt.close(); return null; }
			int pid = rs.getInt(1);
			td.name = rs.getString(2);
			td.email = rs.getString(3);
			td.team_name = rs.getString(4);
			td.pid = pid;

			rs = stmt.executeQuery(
				" select attrib, value from SEM_SPEC_PARTICIPANT_ATTRIB_VL " +
				" where event_id = '" + event_id + "' and " +
				" subevent_id = '" + subevent_id + "' and " +
				" pid = " + pid);
			
			while (rs.next()) {
				td.insertSEMSpecPVals(rs.getString(1), rs.getString(2));
			}

			rs = stmt.executeQuery(
				" select name, team_member.email_id " +
				" from Team_Member, Team_Member_Of " +
				" where Team_Member_Of.pid = " + pid + 
				" and Team_Member_Of.email_id = Team_Member.email_id");
			while (rs.next()) {
				td.insertMember(rs.getString(1), rs.getString(2));
			}
			
			rs.close();
			stmt.close();
			return td;
		} catch (SQLException sqle) {
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return null;
	}

	/**
	 * generates a <code>Vector</code> object for the given sub-event.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id	the id of the subevent
	 * @return a <code>Vector</code> object for this subevent
	 */
	public synchronized Vector getResults(String event_id, String subevent_id) throws IllegalArgumentException {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery(
				" select description, result from Results where event_id = '" + event_id + "' and subevent_id = '" + subevent_id + "'");

			Vector ret = new Vector(20, 20);

			while (rs.next()) {
				ret.addElement(rs.getString(1));
				ret.addElement(rs.getString(2));
			}

			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}

		return null;
	}

	/**
	 * gets a participant pid.
	 *
	 * @param login_id the login ID
	 * @param name the name of the participant
	 * @param tb a flag: true if team-based, false otherwise
	 * @return the pid of the participant
	 */
	public synchronized int getPID(String login_id, String name, boolean tb) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs;
			if (tb) {
				rs = stmt.executeQuery(
					" select pid from Team where team_name = '" + name + "' and login_id = '" + login_id + "'");
			} else {
				rs = stmt.executeQuery(
					" select pid from Individual where name = '" + name + "' and login_id = '" + login_id + "'");
			}

			if (! rs.next()) { rs.close(); stmt.close(); return -1; }
			int pid = rs.getInt(1);
			rs.close();
			stmt.close();
			return pid;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
		return -1;
	}
			
	/**
	 * find user type.
	 *
	 * @param	userid	the login id
	 * @return 0 for event manager <br> 1 for subevent managers, <br> 2 for individual user, and <br>3 for contingents
	 */
	public synchronized int getUserType(String userid) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery("select * from Event_Manager where login_id = '" +
				userid + "'");
			if (rs.next()) { rs.close(); stmt.close(); return 0; }
			rs = stmt.executeQuery("select * from SubEvent_Manager where login_id = '" +
				userid + "'");
			if (rs.next()) { rs.close(); stmt.close(); return 1; }
			rs = stmt.executeQuery("select * from Individual where login_id = '" +
				userid + "'");
			if (rs.next()) { rs.close(); stmt.close(); return 2; }
			rs = stmt.executeQuery("select * from Contingent where login_id = '" +
				userid + "'");
			if (rs.next()) { rs.close(); stmt.close(); return 3; }
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
		return 4;
	}
	
	/**
	 * gets the event ID for an Event Manager/SubEvent Manager.
	 *
	 * @param login_id the login ID of the flo
	 * @param type 0 for Event Manager, 1 for SubEvent Manager
	 * @return the event ID
	 */
	public synchronized String getEventID(String login_id, int type) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs;

			if (type == 0) {
				rs = stmt.executeQuery(
					" select event_id from Event_Manager where login_id = '" + login_id + "'");
			} else {
				rs = stmt.executeQuery(
					" select event_id from SubEvent_Manager where login_id = '" + login_id + "'");
			}

			if (! rs.next()) { rs.close(); stmt.close(); return null; }
			String ret = rs.getString(1);
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}

		return null;
	}
			
	/**
	 * returns subevent id for the given SubEvent Manager login id
	 *
	 * @param userid the SubEvent Manager login id
	 * @return the SubEvent id
	 */
	public synchronized String getSubEventID(String userid) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery("select subevent_id from SubEvent_Manager where "
				+ "login_id = '" + userid + "'");
			String ret = (rs.next() ? rs.getString(1) : null);
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}

		return null;
	}

	/**
	 * returns a <code>SubEventData</code> object which contains data for
	 * the specified SubEvent.
	 * 
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the SubEvent
	 * @return a <code>SubEventData</code> object
	 */
	public synchronized SubEventData getSubEventData(String event_id, String subevent_id) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery(
				" select * from SubEvent where event_id = '" + event_id + "'" +
				" and subevent_id = '" + subevent_id + "'");
			if (! rs.next()) { rs.close(); stmt.close(); return null; }
			SubEventData sed = new SubEventData();
			sed.event_id = rs.getString(1);
			sed.subevent_id = rs.getString(2);
			sed.subevent_name = rs.getString(3);
			sed.venue = rs.getString(4);
			sed.capacity = rs.getInt(5);
			sed.moderation_flag = rs.getInt(6);
			sed.team_size = rs.getInt(7);
			
			rs = stmt.executeQuery(
				" select day, start_time, end_time " +
				" from Occurs_On " +
				" where event_id = '" + event_id + "' and subevent_id = '" + subevent_id + "'");
			
			while (rs.next()) {
				sed.insertTimeInfo(rs.getString(1), rs.getString(2), rs.getString(3));
			}
			
			rs = stmt.executeQuery(
				" select attrib, value " +
				" from EM_SPEC_SUBEVENT_ATTRIB_VAL " +
				" where event_id = '" + event_id + "' and subevent_id = '" + subevent_id + "'");
			
			while (rs.next()) {
				sed.insertAttribVal(rs.getString(1), rs.getString(2));
			}
			
			rs.close();
			stmt.close();
			return sed;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}

		return null;
	}
			
	/** returns a <code>Vector</code> of <code>Attrib</code>s which are required for
	 * registration, apart from the fixed (default) attributes such as email_id,
	 * etc.
	 *
	 * @param event_id the ID of the event
	 * @return a <code>Vector</code> of <code>Attrib</code>s
	 */
	public synchronized Vector getEMSpecPAttribs(String event_id) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery("select attrib, mandatory_flag from EM_SPEC_PARTICIPANT_ATTRIB " +
				" where event_id = '" + event_id + "'");

			Vector ret = new Vector(10, 10);
			boolean flag;
			while (rs.next()) {
				 flag = (rs.getInt(2) == 1) ? true : false;
				 ret.addElement(new Attrib(rs.getString(1), flag));
			}
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return null;
	}
	
	/** returns a <code>Vector</code> of participant <code>Attrib</code>s which are
	 * required to be filled by the subevent manager.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @return a <code>Vector</code> of <code>Attrib</code>s
	 */
	public synchronized Vector getSEMFilledPAttribs(String event_id, String subevent_id) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery(
				" select attrib, mandatory_flag from SEM_SPEC_PARTICIPANT_ATTRIB " +
				" where event_id = '" + event_id + "' " +
				" and subevent_id = '" + subevent_id + "' and sem_filled = 1");

			Vector ret = new Vector(10, 10);
			boolean flag;
			while (rs.next()) {
				 flag = (rs.getInt(2) == 1) ? true : false;
				 ret.addElement(new Attrib(rs.getString(1), flag));
			}
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return null;
	}
	
	/** returns a <code>Vector</code> of <code>Attrib</code>s holding
	 * the attrib-value pairs for a given subevent, for a given 
	 * Participant - only the SEM filled ones.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @param login_id the login ID of the Individual or the Contingent
	 * @param name the name of the Participant
	 * @return a <code>Vector</code> of <code>Attrib</code>s
	 */
	public synchronized Vector getSEMFilledPAttribVals(String event_id, String subevent_id, String login_id, String name) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs;
			boolean IC_flag = SharedObjects.val.isTeamBased(event_id);
			int pid = getPID(login_id, name, IC_flag);

			rs = stmt.executeQuery(
				" select SEM_SPEC_PARTICIPANT_ATTRIB_VL.attrib, SEM_SPEC_PARTICIPANT_ATTRIB_VL.value " + 
				" from SEM_SPEC_PARTICIPANT_ATTRIB_VL, SEM_SPEC_PARTICIPANT_ATTRIB " +
				" where SEM_SPEC_PARTICIPANT_ATTRIB_VL.event_id = '" + event_id + "' and SEM_SPEC_PARTICIPANT_ATTRIB_VL.subevent_id = '" + subevent_id + "'  and " +
				" SEM_SPEC_PARTICIPANT_ATTRIB.event_id = '" + event_id + "' and " +
				" SEM_SPEC_PARTICIPANT_ATTRIB.subevent_id = '" + subevent_id + "' and " +
				" SEM_SPEC_PARTICIPANT_ATTRIB_VL.attrib = SEM_SPEC_PARTICIPANT_ATTRIB.attrib and " +
				" SEM_SPEC_PARTICIPANT_ATTRIB.sem_filled = 1 and " +
				" SEM_SPEC_PARTICIPANT_ATTRIB_VL.pid = " + pid);
			Vector ret = new Vector(10, 10);
			Attrib a;
			while (rs.next()) {
				a = new Attrib(rs.getString(1), false);
				a.value = rs.getString(2);
				ret.addElement(a);
			}
			
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return null;
	}
	
	/** returns a <code>Vector</code> of <code>Attrib</code>s holding
	 * the attrib-value pairs for a given subevent, for a given 
	 * Participant - only the Participant filled ones.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @param login_id the login ID of the Individual or the Contingent
	 * @param name the name of the Participant
	 * @return a <code>Vector</code> of <code>Attrib</code>s
	 */
	public synchronized Vector getSEMSpecPAttribVals(String event_id, String subevent_id, String login_id, String name) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs;
			boolean IC_flag = SharedObjects.val.isTeamBased(event_id);
			int pid = getPID(login_id, name, IC_flag);

			rs = stmt.executeQuery(
				" select SEM_SPEC_PARTICIPANT_ATTRIB_VL.attrib, SEM_SPEC_PARTICIPANT_ATTRIB_VL.value " + 
				" from SEM_SPEC_PARTICIPANT_ATTRIB_VL, SEM_SPEC_PARTICIPANT_ATTRIB " +
				" where SEM_SPEC_PARTICIPANT_ATTRIB_VL.event_id = '" + event_id + "' and SEM_SPEC_PARTICIPANT_ATTRIB_VL.subevent_id = '" + subevent_id + "'  and " +
				" SEM_SPEC_PARTICIPANT_ATTRIB.event_id = '" + event_id + "' and " +
				" SEM_SPEC_PARTICIPANT_ATTRIB.subevent_id = '" + subevent_id + "' and " +
				" SEM_SPEC_PARTICIPANT_ATTRIB_VL.attrib = SEM_SPEC_PARTICIPANT_ATTRIB.attrib and " +
				" SEM_SPEC_PARTICIPANT_ATTRIB.sem_filled = 0 and " +
				" SEM_SPEC_PARTICIPANT_ATTRIB_VL.pid = " + pid);
			Vector ret = new Vector(10, 10);
			Attrib a;
			while (rs.next()) {
				a = new Attrib(rs.getString(1), false);
				a.value = rs.getString(2);
				ret.addElement(a);
			}
			
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return null;
	}
	
	/** returns a <code>Vector</code> of <code>Attrib</code>s which are required to be
	 * filled by a SubEvent Manager for a SubEvent, apart from the fixed (default)
	 * attributes such as subevent_name, venue, capacity etc.
	 *
	 * @param event_id the ID of the event
	 * @return a <code>Vector</code> of <code>Attrib</code>s
	 */
	public synchronized Vector getEMSpecSEAttribs(String event_id) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery("select attrib, mandatory_flag from EM_SPEC_SUBEVENT_ATTRIB " +
				" where event_id = '" + event_id + "'");
			
			Vector ret = new Vector(10, 10);
			boolean flag;
			while (rs.next()) {
				 flag = (rs.getInt(2) == 1) ? true : false;
				 ret.addElement(new Attrib(rs.getString(1), flag));
			}
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return null;
	}
	
	/** returns a <code>Vector</code> of <code>Attrib</code>s which are required for
	 * participation in a subevent, apart from the fixed (default) attributes.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent_id
	 * @return a <code>Vector</code> of <code>Attrib</code>s
	 */
	public synchronized Vector getSEMSpecPAttribs(String event_id, String subevent_id) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery("select attrib, mandatory_flag from SEM_SPEC_PARTICIPANT_ATTRIB " +
				" where event_id = '" + event_id + "' and subevent_id = '" +
				subevent_id + "' and sem_filled = 0");
			Vector ret = new Vector(10, 10);
			boolean flag;
			while (rs.next()) {
				 flag = (rs.getInt(2) == 1) ? true : false;
				 ret.addElement(new Attrib(rs.getString(1), flag));
			}
			rs.close();
			stmt.close();
			return ret;
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
		return null;
	}
	
	/** returns a <code>ResultSet</code> of subevent_ids and subevent names
	 * for a given event_id.
	 *
	 * @param event_id the ID of the event
	 * @return a <code>ResultSet</code> containing the required info.
	 */
	public synchronized ResultSet getSubEvents(String event_id) {
		try {
			Statement stmt = con.createStatement();
			return stmt.executeQuery("select subevent_id, subevent_name " +
				" from SubEvent where event_id = '" + event_id + "'");
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
		return null;
	}
	
	/** returns a <code>ResultSet</code> of event_ids and event names
	 * handled by the GEM
	 *
	 * @return a <code>ResultSet</code> containing the required info.
	 */
	public synchronized ResultSet getEvents() {
		try {
			Statement stmt = con.createStatement();
			return stmt.executeQuery("select event_id, event_name " +
				" from Event ");
		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
		return null;
	}
	
	/** returns a <code>EMSpecData</code> of event details
	 * for the given event_id
	 *
	 * @param event_id the ID of the event
	 * @return a <code>EMSpecData</code> containing the required info.
	 */
	public synchronized EMSpecData getEventDetails(String event_id) {
		try {
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery("select * " +
				" from Event where event_id = '" + event_id + "'");
			if (! rs.next()) { rs.close(); stmt.close(); return null; }
			EMSpecData esd = new EMSpecData();
			esd.name = rs.getString(2);
			esd.location = rs.getString(3);
			esd.startDate = rs.getDate(4);
			esd.endDate = rs.getDate(5);
			esd.capacity = rs.getInt(6);
			esd.moderated_flag = rs.getInt(7);
			esd.IC_flag = rs.getInt(8);
			rs.close();
			stmt.close();
			return esd;

		} catch (SQLException sqle) {
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
		return null;
	}
	
	/** 
	 * enters event details in the Event relation
	 *
	 * @param event_id the ID of the event
	 * @param esd the data specified by the Event Manager.
	 * @return true on success, false otherwise
	 */
	public synchronized boolean insertEventDetails(String event_id, EMSpecData esd) {
		try {
			con.setAutoCommit(true);
			PreparedStatement ps = con.prepareStatement("update Event " +
				"set event_name = ?, location = ?, IC_flag = ?, " +
				"moderation_flag = ?, start_date = ?, end_date = ? " + 
				"where event_id = ?");
			ps.setString(1, esd.name);
			ps.setString(2, esd.location);
			ps.setInt(3, esd.IC_flag);
			ps.setInt(4, esd.moderated_flag);
			ps.setDate(5, esd.startDate);
			ps.setDate(6, esd.endDate);
			ps.setString(7, event_id);

			ps.executeUpdate();
			ps.close();
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}
	
	/** enters extra Event Manager specified SE attribs into
	 * the EM_Spec_SubEvent_Attrib relation from an array of extra attributes.
	 * This array may have empty strings after some point. The mandatory_flags
	 * are also set.
	 *
	 * @param event_id the ID of the event
	 * @param attribs the array of the extra attributes
	 * @param mand the array of the mandatory_flags
	 * @return true if success, false if not.
	 */
	public synchronized boolean insertEMSpecSEAttribs(String event_id, String[] attribs, int[] mand) {
		try {
			PreparedStatement ps = con.prepareStatement("insert into EM_Spec_SubEvent_Attrib "+
				" values (?, ?, ?)");
			con.setAutoCommit(false);
			ps.setString(1, event_id);
			for (int i = 0; i < attribs.length; i++) {
				if (attribs[i].equals("")) continue;
				ps.setString(2, attribs[i]);
				ps.setInt(3, mand[i]);
				ps.executeUpdate();
			}
			ps.close();
			con.commit();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}
	
	/** enters extra Event Manager specified Participant attribs into
	 * the EM_Spec_Participant_Attrib relation from an array of attribs.
	 * This array may have empty strings after some point.
	 * Also, the mandatory_flags are also set then.
	 *
	 * @param event_id the ID of the event
	 * @param attribs the array of the extra attributes
	 * @param mand the array of mandatory_flags
	 * @return true if success, false if not.
	 */
	public synchronized boolean insertEMSpecPAttribs(String event_id, String[] attribs, int[] mand) {
		try {
			PreparedStatement ps = con.prepareStatement("insert into EM_Spec_Participant_Attrib "+
				" values (?, ?, ?)");
			con.setAutoCommit(false);
			ps.setString(1, event_id);
			for (int i = 0; i < attribs.length; i++) {
				if (attribs[i].equals("")) continue;
				ps.setString(2, attribs[i]);
				ps.setInt(3, mand[i]);
				ps.executeUpdate();
			}
			ps.close();
			con.commit();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}
	
	/** enters extra SubEvent Manager specified Participant attribs into
	 * the SEM_Spec_Participant_Attrib relation from an array of attribs.
	 * This array may have empty strings after some point. For each attrib,
	 * the corresponding sem_flag array element should be 0 or 1 depending upon
	 * whether this attribute has to be filled by the SubEvent_Manager or not.
	 * Also, the mandatory_flags are also set then.
	 *
	 * @param event_id the ID of the event
	 * @param subvent_id the ID of the subvent
	 * @param attribs the array of the extra attributes
	 * @param sem_flag the array of flags
	 * @param mand the array of mandatory_flags
	 * @return true if success, false if not.
	 */
	public synchronized boolean insertSEMSpecPAttribs(String event_id, String subevent_id, String[] attribs, int[] sem_flag, int[] mand) {
		try {
			PreparedStatement ps = con.prepareStatement("insert into SEM_SPEC_PARTICIPANT_ATTRIB  "+
				" values (?, ?, ?, ?, ?)");
			con.setAutoCommit(false);
			ps.setString(1, event_id);
			ps.setString(2, subevent_id);
			System.out.println("Gobbledygook");
			for (int i = 0; i < attribs.length; i++) {
				System.out.println("attrib: " + attribs[i]);
				if (attribs[i].equals("")) continue;
				ps.setString(3, attribs[i]);
				ps.setInt(4, sem_flag[i]);
				ps.setInt(5, mand[i]);
				ps.executeUpdate();
			}
			ps.close();
			con.commit();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}

	/**
	 * inserts a new results tuple for a given subevent.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @param desc the description of the result
	 * @param result the value for the result
	 * @return true if success, false otherwise
	 */
	public synchronized boolean insertResult(String event_id, String subevent_id, String desc, String result) {
		try {
			con.setAutoCommit(true);
			Statement stmt = con.createStatement();
			stmt.executeUpdate(
				" insert into Results values('" + event_id + "', '" + subevent_id + "', '" + desc + "', '" + result + "')");
			stmt.close();
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}

	/**
	 * inserts Participant default registration data into either the
	 * Individual or Contingent table.
	 *
	 * @param pdd a <code>PartDefData</code> object
	 * @return true if success, false if not.
	 */
	public synchronized boolean insertPartDefData(PartDefData pdd) {
		try {
			con.setAutoCommit(false);
			Statement stmt = con.createStatement();
			stmt.executeUpdate("insert into Passwords " +
				" values ('" + pdd.login_id + "', '" + pdd.password + "')");
			int flag = 1;
			if (SharedObjects.val.isModerated(pdd.event_id)) flag = 0;
			stmt.executeUpdate("insert into Registered_for " +
				" values ('" + pdd.event_id + "', '" + pdd.login_id + "', " + flag + ")");
			
			if (SharedObjects.val.isTeamBased(pdd.event_id)) {
				stmt.executeUpdate("insert into Contingent values ('" +
					pdd.login_id + "', '" + pdd.name + "', '" + pdd.email_id +
					"', '" + pdd.cl_name + "')");
				stmt.close();
				con.commit();
				con.setAutoCommit(true);
				return true;
			} else {
				int pid = SharedObjects.val.nextPID();
				if (pid == -1) {stmt.close(); con.setAutoCommit(true); return false; }
				stmt.executeUpdate("insert into Participants values (" + pid +
					", '" + pdd.name + "')");
				stmt.executeUpdate("insert into Individual values (" + pid + ", '" +
					pdd.email_id + "', '" + pdd.login_id + "', '" + pdd.name +
					"')");
				stmt.close();
				con.commit();
				con.setAutoCommit(true);
				return true;
			}
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}

	/**
	 * inserts the team members for a contingent into the Team_Member,
	 * Team and Team_Member_Of tables.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @param login_id the login ID of the contingent
	 * @param names an array of the names of the team members
	 * @param emails an array of the emails of the team members
	 * @param team_name the name of the team
	 * @param name the contact name
	 * @param email the contact email ID
	 * @return true if success, false otherwise
	 */
	public synchronized boolean insertTeamEntry(String event_id, String subevent_id, String login_id, String[] names, String[] emails, String team_name, String name, String email) {
		try {
			con.setAutoCommit(false);
			Statement stmt = con.createStatement();
			int pid = SharedObjects.val.nextPID();
			if (pid == -1) {stmt.close(); con.setAutoCommit(true); return false; }
			stmt.executeUpdate("insert into Participants values (" + pid +
				", '" + team_name + "')");
			
			stmt.executeUpdate("insert into Team values (" + pid + ", '" +
				login_id + "', '" + team_name + "', '" + name + "', '" + email +
				"')");
			stmt.close();

			PreparedStatement ps1 = con.prepareStatement(
				" insert into Team_Member values(?, ?)");
			PreparedStatement ps2 = con.prepareStatement(
				" insert into Team_Member_Of values(" + pid + ", ?)");
			
			for (int i = 0; i < emails.length; i++) {
				emails[i] = emails[i].trim();
				names[i] = names[i].trim();
				if (emails[i].equals("") || names[i].equals("")) { ps1.close(); ps2.close(); return false; }
				if (! SharedObjects.val.isTeamMember(emails[i])) {
					ps1.setString(1, emails[i]);
					ps1.setString(2, names[i]);
					ps1.executeUpdate();
				}
				ps2.setString(1, emails[i]);
				ps2.executeUpdate();
			}
			ps1.close();
			ps2.close();
			con.commit();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}
					  
	/**
	 * inserts the Participant details into SEM_SPEC_PARTICIPANT_ATTRIB_VAL.
	 * Also registers the participant for the subevent by adding an entry
	 * in the Participates_In table.
	 * This also checks if any mandatory fields are not filled.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @param userid the login ID of the Participant
	 * @param name the name of the participant, the team name if team-based
	 * @param vec A <code>Vector</code> of <code>Attrib<code>
	 * @return true if success, false if not.
	 */
	public synchronized boolean insertSEMSpecPVals(String event_id, String subevent_id, String userid, String name, Vector vec) {
		Enumeration e = vec.elements();
		try {
			name = name.trim();
			Statement stmt = con.createStatement();
			ResultSet rs;
			boolean IC_flag = SharedObjects.val.isTeamBased(event_id);
			int mod_flag = (SharedObjects.val.isModerated(event_id, subevent_id)) ? 0 : 1;

			if (IC_flag) {
				rs = stmt.executeQuery(
					"select pid from Team where login_id = '" + userid + "' and team_name = '" + name + "'");
			} else {
				rs = stmt.executeQuery(
					"select pid from Individual where login_id = '" + userid + "' and name = '" + name + "'");
			}

			if (! rs.next()) { rs.close(); stmt.close(); return false; }
			int pid = rs.getInt(1);
			rs.close();
			
			Attrib a;
			String a_name;
			String a_value;
			con.setAutoCommit(false);
			
			stmt.executeUpdate(
				"insert into Participates_In values ('" + event_id + "', '" + subevent_id + "', " + pid + ", " + mod_flag + ")");
			
			PreparedStatement ps = con.prepareStatement("insert into " +
				" SEM_SPEC_PARTICIPANT_ATTRIB_VL values(?, ?, ?, ?, ?)");
			ps.setString(1, event_id);
			ps.setString(2, subevent_id);
			ps.setInt(3, pid);
			while (e.hasMoreElements()) {
				a = (Attrib)(e.nextElement());
				a_name = a.name.trim();
				a_value = a.value.trim();
				if ((! a.mand_flag) && a_value.equals("")) {
					con.rollback();
					ps.close();
					con.setAutoCommit(true);
					return false;
				}
				ps.setString(4, a_name);
				ps.setString(5, a_value);
				ps.executeUpdate();
			}
			e = getSEMFilledPAttribs(event_id, subevent_id).elements();
			while (e.hasMoreElements()) {
				a = (Attrib)(e.nextElement());
				a_name = a.name.trim();
				ps.setString(4, a_name);
				ps.setString(5, "");
				ps.executeUpdate();
			}
			
			ps.close();
			con.commit();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}
	
	/**
	 * inserts the Subevent details into EM_SPEC_SUBEVENT_ATTRIB
	 * This also checks if any mandatory fields are not filled.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @param vec A <code>Vector</code> of <code>Attrib<code>
	 * @return true if success, false if not.
	 */
	public synchronized boolean insertEMSpecSEVals(String event_id, String subevent_id, Vector vec) {
		Enumeration e = vec.elements();
		try {
			PreparedStatement ps = con.prepareStatement("insert into " +
				" EM_SPEC_SUBEVENT_ATTRIB_VAL values(?, ?, ?, ?)");
			Attrib a;
			String name;
			String value;
			con.setAutoCommit(false);
			ps.setString(1, event_id);
			ps.setString(2, subevent_id);
			while (e.hasMoreElements()) {
				a = (Attrib)(e.nextElement());
				name = a.name.trim();
				value = a.value.trim();
				if ((! a.mand_flag) && value.equals("")) {
					con.rollback();
					ps.close();
					con.setAutoCommit(true);
					return false;
				}
				ps.setString(3, name);
				ps.setString(4, value);
				ps.executeUpdate();
			}
			ps.close();
			con.commit();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}
	
	/**
	 * inserts the Participant details into EM_SPEC_PARTICIPANT_ATTRIB_VAL
	 * This also checks if any mandatory fields are not filled.
	 *
	 * @param event_id the ID of the event
	 * @param	userid	the login id
	 * @param vec A <code>Vector</code> of <code>Attrib<code>
	 * @return true if success, false if not.
	 */
	public synchronized boolean insertEMSpecPVals(String event_id, String userid, Vector vec) {
		Enumeration e = vec.elements();
		try {
			PreparedStatement ps = con.prepareStatement("insert into " +
				" EM_SPEC_PARTICIPANT_ATTRIB_VAL values(?, ?, ?, ?)");
			Attrib a;
			String name;
			String value;
			con.setAutoCommit(false);
			ps.setString(1, event_id);
			ps.setString(2, userid);
			while (e.hasMoreElements()) {
				a = (Attrib)(e.nextElement());
				name = a.name;
				value = a.value.trim();
				if ((! a.mand_flag) && value.equals("")) {
					con.rollback();
					ps.close();
					con.setAutoCommit(true);
					return false;
				}
				ps.setString(3, name);
				ps.setString(4, value);
				ps.executeUpdate();
			}
			ps.close();
			con.commit();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}

	/**
	 * creates a new entry for an event in the Event table.
	 * This is to be used by the sysad to register an event. This
	 * populates the table with some junk default values.
	 *
	 * @param event_id the ID assigned to this event by the sysad
	 * @param event_name the name of the event
	 * @param em_name the name of the event manager for this event
	 * @param em_login the login ID of the event manager for this event
	 * @param em_passwd the password of the event manager for this event
	 * @param em_email the email ID of the event manager
	 * @param capacity the capacity constraints for this event
	 * @return true if success, false if not.
	 */
	public synchronized boolean insertEventEntry(String event_id, String event_name, String em_name, String em_login, String em_passwd, String em_email, int capacity) {
		try {
			con.setAutoCommit(false);
			Statement stmt = con.createStatement();
			stmt.executeUpdate("insert into Event values ('" + event_id +
				"', '" + event_name + "', 'venue unknown', '01-JAN-2000', '01-JAN-2000'," + capacity + ", 2, 2)");
			stmt.executeUpdate(
				"insert into Passwords values ('" + em_login + "', '" + em_passwd + "')");
			stmt.executeUpdate(
				"insert into Event_Manager values ('" + em_login + "', '" + event_id + "', '" + em_name + "', '" + em_email + "')");
			con.commit();
			stmt.close();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}
	
	/**
	 * creates a new entry for a subevent under an event in the SubEvent table.
	 * This is to be used by the Event Manager to register an event. This
	 * populates the table with some junk default values.
	 *
	 * @param event_id the ID assigned to the event
	 * @param subevent_id the ID assigned to this subevent by the Event Manager
	 * @param subevent_name the name of this subevent
	 * @param sem_name the name of the subevent manager for this subevent
	 * @param sem_login the login ID of this subevent manager
	 * @param sem_passwd the password of this subevent manager
	 * @param sem_email the email ID of this subevent manager
	 * @param capacity the capacity constraints for this subevent
	 * @return true if success, false if not.
	 */
	public synchronized boolean insertSubEventEntry(String event_id, String subevent_id, String subevent_name, String sem_name, String sem_login, String sem_passwd, String sem_email, int capacity) {
		try {
			con.setAutoCommit(false);
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery(
				"select capacity from Event where event_id = '" + event_id + "'");
			if (! rs.next()) {rs.close(); stmt.close(); return false;}
			int max = rs.getInt(1);
			rs.close();
			if (capacity > max) { stmt.close(); return false; }	
			stmt.executeUpdate("insert into SubEvent values ('" + event_id +
				"', '" + subevent_id + "', '" + subevent_name + "', 'venue unknown', " + capacity + ", 2, 1)");
			stmt.executeUpdate(
				"insert into Passwords values ('" + sem_login + "', '" + sem_passwd + "')");
			stmt.executeUpdate(
				"insert into SubEvent_Manager values ('" + sem_login + "', '" + event_id + "', '" + subevent_id + "', '" + sem_name + "', '" + sem_email + "')");
			con.commit();
			stmt.close();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}

	/**
	 * inserts a Judge for a SubEvent.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the SubEvent
	 * @param j a <code>Judge</code> object
	 * @return true if success, false if not
	 */
	public synchronized boolean insertJudge(String event_id, String subevent_id, Judge j) {
		try {
			con.setAutoCommit(false);
			PreparedStatement ps = con.prepareStatement(
				"insert into Judge values(?, ?, ?, ?, ?, ?)");
			ps.setString(1, j.email_id);
			ps.setString(2, j.name);
			ps.setString(3, j.phone);
			ps.setString(4, j.address1);
			ps.setString(5, j.address2);
			ps.setString(6, j.fax);
			ps.executeUpdate();
			ps = con.prepareStatement(
				"insert into Judges values(?, ?, ?)");
			ps.setString(1, event_id);
			ps.setString(2, subevent_id);
			ps.setString(3, j.email_id);
			ps.executeUpdate();
			con.commit();
			ps.close();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}

	/**
	 * deletes a result tuple
	 * 
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @param desc the description of the result
	 * @param result the value for the result
	 * @return true if success, false otherwise
	 */
	public synchronized boolean deleteResult(String event_id, String subevent_id, String desc, String result) {
		try {
			con.setAutoCommit(true);
			Statement stmt = con.createStatement();
			stmt.executeUpdate(
				" delete from Results where " +
				" event_id = '" + event_id + "' and " +
				" subevent_id = '" + subevent_id + "' and " +
				" description = '" + desc + "' and " +
				" result = '" + result + "'");
			stmt.close();
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception " + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}

	/**
	 * deletes a whole event!!!
	 * Use with extreme caution. To be used by sysad wonly!!
	 *
	 * @param event_id the id of the event to be khatched.
	 * @return true if success, false otherwise
	 */
	public synchronized boolean deleteEvent(String event_id) {
		try {
			boolean IC_flag = SharedObjects.val.isTeamBased(event_id);
			Statement stmt = con.createStatement();
			
			con.setAutoCommit(false);
			if (IC_flag) {
				stmt.executeUpdate(
				" delete from Team_Member " +
				" where email_id in " +
				" (select team_Member_of.email_id " +
				" from Team, Team_Member_of, Participates_In" +
				" where team.pid = Team_Member_of.pid and " +
				" team.pid = Participates_In.pid and " +
				" Participates_In.event_id = '" + event_id + "')");
			}
			stmt.executeUpdate(
				"delete from Passwords " +
				" where login_id in (select login_id from registered_for " +
				" where event_id = '" + event_id + "')");
			stmt.executeUpdate(
				"delete from Passwords " +
				" where login_id in (select login_id from Event_Manager " +
				" where event_id = '" + event_id + "')");
			stmt.executeUpdate(
				"delete from Passwords " +
				" where login_id in (select login_id from SubEvent_Manager " +
				" where event_id = '" + event_id + "')");
			stmt.executeUpdate(
				" delete from Judge " +
				" where email_id in (select email_id from Judges " +
				" where event_id = '" + event_id + "')");
			stmt.executeUpdate(
				"delete from Event " +
				" where event_id = '" + event_id + "'");
			con.commit();
			stmt.close();
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}

		return reorder();
	}
		
	/** reorders the pids after a delete
	 *
	 * @return true if success, false otherwise
	 */
	public synchronized boolean reorder() {
		try {
			/** <b>the following stuff should preferrably be in
			 * a trigger </B> */
			Statement stmt1 = con.createStatement();
			stmt1.executeUpdate(
				" create view temp(old, new) as " +
				" select p.pid, count(q.pid) " +
				" from participants p, participants q  " +
				" where p.pid >= q.pid group by p.pid ");
			stmt1.executeUpdate(
				" update Participants " +
				" set pid = (select new from temp " +
				" where Participants.pid = temp.old)");
			stmt1.executeUpdate(
				" update Participates_in " +
				" set pid = (select new from temp " +
				" where Participates_in.pid = temp.old) ");
			stmt1.executeUpdate(
				" update Individual " +
				" set pid = (select new from temp " +
				" where Individual.pid = temp.old) ");
			stmt1.executeUpdate(
				" update Team_Member_of " +
				" set pid = (select new from temp " +
				" where Team_Member_of.pid = temp.old) ");
			stmt1.executeUpdate(
				" update Team " +
				" set pid = (select new from temp " +
				" where Team.pid = temp.old) ");
			stmt1.executeUpdate("drop view temp");
			con.commit();
			stmt1.close();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
				Statement stmt2 = con.createStatement();
				stmt2.executeUpdate("drop view temp");
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}

	/**
	 * deletes a whole subevent!!!
	 * Use with extreme caution. To be used by event manager wonly!!
	 *
	 * @param event_id the id of the event
	 * @param subevent_id the ID of the subevent
	 * @return true if success, false otherwise
	 */
	public synchronized boolean deleteSubEvent(String event_id, String subevent_id) {
		try {
			boolean IC_flag = SharedObjects.val.isTeamBased(event_id);
			Statement stmt = con.createStatement();
			
			con.setAutoCommit(false);
			if (IC_flag) {
				stmt.executeUpdate(
				" delete from Team_Member " +
				" where email_id in " +
				" (select team_Member_of.email_id " +
				" from Team, Team_Member_of, Participates_In" +
				" where team.pid = Team_Member_of.pid and " +
				" team.pid = Participates_In.pid and " +
				" Participates_In.event_id = '" + event_id + "' and " +
				" Participates_In.subevent_id = '" + subevent_id + "')");
			}
			stmt.executeUpdate(
				" delete from Judges " +
				" where event_id = '" + event_id + "' and " +
				" subevent_id = '" + subevent_id + "'");
			stmt.executeUpdate(
				" delete from Judge " +
				" where email_id not in (select email_id from Judges)");
			stmt.executeUpdate(
				"delete from Passwords " +
				" where login_id in (select login_id from SubEvent_Manager " +
				" where event_id = '" + event_id + "' and " +
				" subevent_id = '" + subevent_id + "')");
			stmt.executeUpdate(
				"delete from SubEvent " +
				" where event_id = '" + event_id + "' and " +
				" subevent_id = '" + subevent_id + "'");
			con.commit();
			stmt.close();
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}

		return reorder();
	}

	/** removes a judge for a subevent
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the SubEvent
	 * @param email the email ID of the judge to be removed
	 * @return true on success, false otherwise
	 */
	public synchronized boolean deleteJudge(String event_id, String subevent_id, String email) {
		try {
			con.setAutoCommit(false);
			Statement stmt = con.createStatement();
			stmt.executeUpdate(
				" delete from Judges " +
				" where event_id = '" + event_id + "' and " +
				" subevent_id = '" + subevent_id + "' and " +
				" email_id = '" + email + "'");
			stmt.executeUpdate(
				" delete from Judge " +
				" where email_id not in (select email_id from Judges)");
			con.commit();
			stmt.close();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}
	
	/**
	 * updates the Subevent details in EM_SPEC_SUBEVENT_ATTRIB
	 * This also checks if any mandatory fields are not filled.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @param vec A <code>Vector</code> of <code>Attrib<code>
	 * @return true if success, false if not.
	 */
	public synchronized boolean updateEMSpecSEVals(String event_id, String subevent_id, Vector vec) {
		if (vec == null)
		System.out.println("vec null?");
		Enumeration e = vec.elements();
		System.out.println("vec not null?");
		try {
			PreparedStatement ps = con.prepareStatement(
				"update EM_SPEC_SUBEVENT_ATTRIB_VAL " +
				" set value = ? " + 
				" where event_id = ? and subevent_id = ? and attrib = ?");
			Attrib a;
			String name;
			String value;
			con.setAutoCommit(false);
			ps.setString(2, event_id);
			ps.setString(3, subevent_id);
			System.out.println("size " + vec.size());
			while (e.hasMoreElements()) {
				System.out.println("here boy");
				a = (Attrib)(e.nextElement());
				if (a == null) System.out.println(" a null");
				name = a.name.trim();
				System.out.println(name);
				value = a.value.trim();
				if ((! a.mand_flag) && value.equals("")) {
					con.rollback();
					ps.close();
					con.setAutoCommit(true);
					return false;
				}
				ps.setString(4, name);
				ps.setString(1, value);
				ps.executeUpdate();
			}
			ps.close();
			con.commit();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}
	
	/**
	 * updates the Participant details in EM_SPEC_PARTICIPANT_ATTRIB_VAL
	 * This also checks if any mandatory fields are not filled.
	 *
	 * @param event_id the ID of the event
	 * @param	userid	the login id
	 * @param vec A <code>Vector</code> of <code>Attrib<code>
	 * @return true if success, false if not.
	 */
	public synchronized boolean updateEMSpecPVals(String event_id, String userid, Vector vec) {
		Enumeration e = vec.elements();
		try {
			PreparedStatement ps = con.prepareStatement(
				" update EM_SPEC_PARTICIPANT_ATTRIB_VAL " +
				" set value = ? " + 
				" where event_id = ? and subevent_id = ? and attrib = ?");
			Attrib a;
			String name;
			String value;
			con.setAutoCommit(false);
			ps.setString(2, event_id);
			ps.setString(3, userid);
			while (e.hasMoreElements()) {
				a = (Attrib)(e.nextElement());
				name = a.name;
				value = a.value.trim();
				if ((! a.mand_flag) && value.equals("")) {
					con.rollback();
					ps.close();
					con.setAutoCommit(true);
					return false;
				}
				ps.setString(4, name);
				ps.setString(1, value);
				ps.executeUpdate();
			}
			ps.close();
			con.commit();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}
	
	/**
	 * updates the Participant details in SEM_SPEC_PARTICIPANT_ATTRIB_VAL
	 * This also checks if any mandatory fields are not filled.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @param userid the login ID of the Participant
	 * @param name the name of the participant, the team name if team-based
	 * @param vec A <code>Vector</code> of <code>Attrib<code>
	 * @return true if success, false if not.
	 */
	public synchronized boolean updateSEMSpecPVals(String event_id, String subevent_id, String userid, String name, Vector vec) {
		Enumeration e = vec.elements();
		try {
			name = name.trim();
			Statement stmt = con.createStatement();
			ResultSet rs;
			boolean IC_flag = SharedObjects.val.isTeamBased(event_id);
			if (IC_flag) {
				rs = stmt.executeQuery(
					"select pid from Team where login_id = '" + userid + "' and team_name = '" + name + "'");
			} else {
				rs = stmt.executeQuery(
					"select pid from Individual where login_id = '" + userid + "' and name = '" + name + "'");
			}

			if (! rs.next()) { rs.close(); stmt.close(); return false; }
			int pid = rs.getInt(1);
			rs.close();
			PreparedStatement ps = con.prepareStatement(
				" update SEM_SPEC_PARTICIPANT_ATTRIB_VL " +
				" set value = ? " + 
				" where event_id = ? and subevent_id = ? and pid = ? and attrib = ?");
			Attrib a;
			String a_name;
			String a_value;
			con.setAutoCommit(false);
			ps.setString(2, event_id);
			ps.setString(3, subevent_id);
			ps.setInt(4, pid);
			while (e.hasMoreElements()) {
				a = (Attrib)(e.nextElement());
				a_name = a.name.trim();
				a_value = a.value.trim();
				if ((! a.mand_flag) && a_value.equals("")) {
					con.rollback();
					ps.close();
					con.setAutoCommit(true);
					return false;
				}
				ps.setString(5, a_name);
				ps.setString(1, a_value);
				ps.executeUpdate();
			}
			ps.close();
			con.commit();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}
	
	/**
	 * updates the SEM filled Participant details in SEM_SPEC_PARTICIPANT_ATTRIB_VAL
	 * This also checks if any mandatory fields are not filled.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @param userid the login ID of the Participant
	 * @param name the name of the participant, the team name if team-based
	 * @param vec A <code>Vector</code> of <code>Attrib<code>
	 * @return true if success, false if not.
	 */
	public synchronized boolean updateSEMFilledPVals(String event_id, String subevent_id, String userid, String name, Vector vec) {
		return updateSEMSpecPVals(event_id, subevent_id, userid, name, vec);
	}
	
	/**
	 * updates the SubEvent and Occurs On tables.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @param sed a <code>SubEventData</code> object for this SubEvent
	 * @return true if success, false if not.
	 */
	public synchronized boolean updateSubEventData(String event_id, String subevent_id, SubEventData sed) {
		try {
			con.setAutoCommit(false);
			Statement stmt = con.createStatement();
			ResultSet rs = stmt.executeQuery(
				"select capacity from Event where event_id = '" + event_id + "'");
			if (! rs.next()) {rs.close(); stmt.close(); return false;}
			int max = rs.getInt(1);
			rs.close();
			if (sed.capacity > max) { stmt.close(); return false; }	
			
			stmt.executeUpdate(
				" update SubEvent " + 
				" set subevent_name = '" + sed.subevent_name + "',  venue = '" + sed.venue + "', moderation_flag = " + sed.moderation_flag + ", team_size = " + sed.team_size +
				" where event_id = '" + event_id + "' and subevent_id = '" + subevent_id + "'");
			updateEMSpecSEVals(event_id, subevent_id, sed.attrib_vals);

			stmt.executeUpdate(
				" delete from Occurs_On " +
				" where event_id = '" + event_id + "' and subevent_id = '" + subevent_id + "'");
			stmt.close();

			PreparedStatement ps = con.prepareStatement(
				" insert into Occurs_On values (?, ?, to_date(?, 'YYYY-MM-DD'), to_date(?, 'HH24:MI'), to_date(?, 'HH24:MI') )");
			
			ps.setString(1, event_id);
			ps.setString(2, subevent_id);
			Enumeration e = sed.times.elements();
			TimeInfo ti;
			while (e.hasMoreElements()) {
				ti = (TimeInfo)(e.nextElement());
				ps.setString(3, ti.date);
				ps.setString(4, ti.start_time);
				ps.setString(5, ti.end_time);
				ps.executeUpdate();
			}
			ps.close(); 
			con.commit();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}
	
	/**
	 * updates Judge info for a SubEvent.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the SubEvent
	 * @param j a <code>Judge</code> object
	 * @return true if success, false if not
	 */
	public synchronized boolean updateJudge(String event_id, String subevent_id, Judge j) {
		try {
			con.setAutoCommit(false);
			PreparedStatement ps = con.prepareStatement(
				" update Judge set name = ?, phone = ?, " +
				" address1 = ?, address2 = ?, fax = ? " + 
				" where email_id = ?");
			ps.setString(6, j.email_id);
			ps.setString(1, j.name);
			ps.setString(2, j.phone);
			ps.setString(3, j.address1);
			ps.setString(4, j.address2);
			ps.setString(5, j.fax);
			ps.executeUpdate();
			con.commit();
			ps.close();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}

	/**
	 * updates the pending registration of an Individual/Contingent.
	 * 
	 * @param event_id The ID of the event
	 * @param v The <code>Vector</code> of registrant info
	 * @return true if success, false otherwise
	 */	
	public synchronized boolean updatePending(String event_id, Vector v) {
		try {
			con.setAutoCommit(false);
			PreparedStatement ps1 = con.prepareStatement(
				" update registered_for set approved = 1 where login_id = ? and event_id = ?");
			PreparedStatement ps2 = con.prepareStatement(
				" delete from registered_for where login_id = ? and event_id = ?");
			ps1.setString(2, event_id);
			ps2.setString(2, event_id);
			Enumeration e = v.elements();
			String login_id, name;
			int approved;
			while (e.hasMoreElements()) {
				login_id = (String)(e.nextElement());
				name = (String)(e.nextElement());
				approved = ((Integer)(e.nextElement())).intValue();

				if (approved == 1) {
					ps1.setString(1, login_id);
					ps1.executeUpdate();
				} else {
					ps2.setString(1, login_id);
					ps2.executeUpdate();
				}
			}

			ps1.close();
			ps2.close();
			con.setAutoCommit(true);
			con.commit();
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}

		return reorder();
	}
	
	/**
	 * updates the pending participation of an Individual/Team in a
	 * subevent.
	 * 
	 * @param event_id The ID of the event
	 * @param subevent_id the ID of the subevent
	 * @param v The <code>Vector</code> of registrant info
	 * @return true if success, false otherwise
	 */
	public synchronized boolean updatePending(String event_id, String subevent_id, Vector v) {
		try {
			con.setAutoCommit(false);
			boolean tb = SharedObjects.val.isTeamBased(event_id);
			PreparedStatement ps1 = con.prepareStatement(
				" update participates_in set approved = 1 where pid = ? and event_id = ? and subevent_id = ?");
			PreparedStatement ps2 = con.prepareStatement(
				" delete from participates_in where pid = ? and event_id = ? and subevent_id = ?");
			ps1.setString(2, event_id);
			ps1.setString(3, subevent_id);
			ps2.setString(2, event_id);
			ps2.setString(3, subevent_id);
			Enumeration e = v.elements();
			String login_id, name;
			int approved, pid;
			while (e.hasMoreElements()) {
				login_id = (String)(e.nextElement());
				name = (String)(e.nextElement());
				approved = ((Integer)(e.nextElement())).intValue();
			
				pid = getPID(login_id, name, tb);
				System.out.println("name :" + name + ": login :" + login_id + ":");
				System.out.println("pid " + pid + " approved " + approved);
				if (approved == 1) {
					ps1.setInt(1, pid);
					ps1.executeUpdate();
				} else {
					ps2.setInt(1, pid);
					ps2.executeUpdate();
				}
			}

			ps1.close();
			ps2.close();
			con.setAutoCommit(true);
			con.commit();
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}

		return reorder();
	}

	/**
	 * updates a result tuple in the Results table.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @param odesc the old description of the result
	 * @param ores the old value for the result
	 * @param ndesc the old description of the result
	 * @param nres the old value for the result
	 * @return true if success, false otherwise
	 */
	public synchronized boolean updateResult(String event_id, String subevent_id, String odesc, String ores, String ndesc, String nres) {
		try {
			con.setAutoCommit(true);
			PreparedStatement ps = con.prepareStatement(
				" update Results set description = ?, result = ? " +
				" where event_id = ? and subevent_id = ? and description = ? and result = ?");
			ps.setString(1, ndesc);
			ps.setString(2, nres);
			ps.setString(3, event_id);
			ps.setString(4, subevent_id);
			ps.setString(5, odesc);
			ps.setString(6, ores);

			ps.executeUpdate();
			ps.close();
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}
	
	/**
	 * updates the team members for a contingent in the Team_Member,
	 * Team and Team_Member_Of tables.
	 *
	 * @param event_id the ID of the event
	 * @param subevent_id the ID of the subevent
	 * @param login_id the login ID of the contingent
	 * @param names an array of the names of the team members
	 * @param emails an array of the emails of the team members
	 * @param team_name the name of the team
	 * @param name the contact name
	 * @param email the contact email ID
	 * @return true if success, false otherwise
	 */
	public synchronized boolean updateTeamEntry(String event_id, String subevent_id, String login_id, String[] names, String[] emails, String team_name, String name, String email) {
		try {
			con.setAutoCommit(false);
			int pid = getPID(login_id, team_name, true);
			Statement stmt = con.createStatement();
			stmt.executeUpdate(
				" delete from Team_Member_Of where pid = " + pid);
			stmt.executeUpdate(
				" delete from Team_Member where email_id not in (select email_id from Team_Member_Of)");
			stmt.executeUpdate(
				" update Team set " +
				" team_name = '" + team_name + "', name = '" + name + "', email_id = '" + email + "' " + 
				" where pid = " + pid);
			
			stmt.close();
			PreparedStatement ps1 = con.prepareStatement(
				" insert into Team_Member values(?, ?)");
			PreparedStatement ps2 = con.prepareStatement(
				" insert into Team_Member_Of values(" + pid + ", ?)");
			
			for (int i = 0; i < emails.length; i++) {
				emails[i] = emails[i].trim();
				names[i] = names[i].trim();
				if (emails[i].equals("") || names[i].equals("")) { ps1.close(); ps2.close(); return false; }
				if (! SharedObjects.val.isTeamMember(emails[i])) {
					ps1.setString(1, emails[i]);
					ps1.setString(2, names[i]);
					ps1.executeUpdate();
				}
				ps2.setString(1, emails[i]);
				ps2.executeUpdate();
			}
			ps1.close();
			ps2.close();
			con.commit();
			con.setAutoCommit(true);
			return true;
		} catch (SQLException sqle) {
			try {
				con.rollback();
			} catch (SQLException sqle1) { sqle1.printStackTrace(); }
			System.out.println("Exception here" + sqle.getMessage());
			sqle.printStackTrace();
		}
		return false;
	}

}
